// Photography Booking Website - Version 3 (HTML/CSS/JS)

const styles = [
  { 
    id: 'portrait', 
    name: 'Портрет', 
    icon: 'icons/style-portrait.svg',
    color: '#FF6B9D',
    description: 'Индивидуальные портретные съёмки',
    duration: '1-2 часа',
    price: 'от 5000₽'
  },
  { 
    id: 'family', 
    name: 'Семейная', 
    icon: 'icons/style-family.svg',
    color: '#C44569',
    description: 'Семейные фотосессии',
    duration: '2-3 часа',
    price: 'от 8000₽'
  },
  { 
    id: 'lovestory', 
    name: 'Love Story', 
    icon: 'icons/style-lovestory.svg',
    color: '#FFB6C1',
    description: 'Романтические съёмки',
    duration: '2-4 часа',
    price: 'от 10000₽'
  },
  { 
    id: 'brand', 
    name: 'Бренд', 
    icon: 'icons/style-brand.svg',
    color: '#9B59B6',
    description: 'Брендовые фотосессии',
    duration: '3-5 часов',
    price: 'от 15000₽'
  }
]

const photographers = [
  { name: 'Анна Петрова', experience: '8 лет', rating: 5, specialty: 'Love Story, Портрет' },
  { name: 'Михаил Соколов', experience: '12 лет', rating: 5, specialty: 'Бренд, Портрет' },
  { name: 'Елена Волкова', experience: '6 лет', rating: 5, specialty: 'Семейная, Love Story' },
  { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Бренд, Портрет' }
]

const advantages = [
  { icon: 'icons/stat-rating.svg', title: 'Профессиональное оборудование', text: 'Работаем только с топовым фотооборудованием' },
  { icon: 'icons/stat-sessions.svg', title: 'Гибкий график', text: 'Подстраиваемся под ваше удобное время' },
  { icon: 'icons/stat-photographer.svg', title: 'Опытные фотографы', text: 'Все наши специалисты с многолетним опытом' },
  { icon: 'icons/form-camera.svg', title: 'Быстрая обработка', text: 'Готовые фото в течение 3-5 дней' }
]

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadSessions()
  updateStats()
  
  setInterval(() => {
    loadSessions()
    updateStats()
  }, 500)
}

function loadSessions() {
  const saved = localStorage.getItem('photoSessions')
  let sessions = []
  
  if (saved) {
    try {
      sessions = JSON.parse(saved)
    } catch (e) {
      sessions = []
    }
  } else {
    sessions = [
      {
        id: 1,
        photographer: 'Анна Петрова',
        theme: 'Love Story',
        date: '2025-03-15',
        rating: 5
      },
      {
        id: 2,
        photographer: 'Михаил Соколов',
        theme: 'Портрет',
        date: '2025-03-10',
        rating: 4
      },
      {
        id: 3,
        photographer: 'Елена Волкова',
        theme: 'Семейная',
        date: '2025-03-05',
        rating: 5
      }
    ]
    localStorage.setItem('photoSessions', JSON.stringify(sessions))
  }
  
  renderSessions(sessions)
  return sessions
}

function getThemeColor(theme) {
  const colors = {
    'Портрет': '#FF6B9D',
    'Семейная': '#C44569',
    'Love Story': '#FFB6C1',
    'Бренд': '#9B59B6'
  }
  return colors[theme] || '#FF6B9D'
}

function renderSessions(sessions) {
  const sessionsGrid = document.getElementById('sessionsGrid')
  if (!sessionsGrid) return
  
  if (sessions.length === 0) {
    sessionsGrid.innerHTML = `
      <div class="empty-state">
        <div class="empty-icon-circle">
          <img src="icons/form-camera.svg" alt="" width="64" height="64" />
        </div>
        <h3 class="empty-title">Нет запланированных съёмок</h3>
        <p class="empty-text">Запишитесь на фотосессию прямо сейчас</p>
      </div>
    `
  } else {
    sessionsGrid.innerHTML = sessions.map(session => {
      const themeColor = getThemeColor(session.theme)
      const stars = Array.from({ length: 5 }, (_, i) => 
        i < session.rating 
          ? `<img src="icons/stat-rating.svg" alt="star" width="20" height="20" style="filter: drop-shadow(0 2px 4px rgba(245, 158, 11, 0.3));" />`
          : `<img src="icons/stat-rating.svg" alt="star" width="20" height="20" style="opacity: 0.3;" />`
      ).join('')
      
      return `
        <div class="session-card" style="border-top: 4px solid ${themeColor};">
          <div class="session-header">
            <div class="session-theme-badge" style="background: ${themeColor}20; color: ${themeColor};">
              ${session.theme}
            </div>
            <div class="session-rating">
              ${stars}
            </div>
          </div>
          <div class="session-content">
            <div class="session-info-item">
              <img src="icons/stat-photographer.svg" alt="" width="20" height="20" style="filter: drop-shadow(0 2px 4px ${themeColor}30);" />
              <span class="session-info-text">${session.photographer}</span>
            </div>
            <div class="session-info-item">
              <img src="icons/form-calendar.svg" alt="" width="20" height="20" style="filter: drop-shadow(0 2px 4px ${themeColor}30);" />
              <span class="session-info-text">
                ${new Date(session.date).toLocaleDateString('ru-RU', { 
                  day: 'numeric', 
                  month: 'long', 
                  year: 'numeric' 
                })}
              </span>
            </div>
          </div>
        </div>
      `
    }).join('')
  }
}

function updateStats() {
  const sessions = loadSessions()
  const stats = [
    { 
      value: sessions.length, 
      label: 'Съёмок',
      icon: 'icons/stat-sessions.svg',
      color: '#FF6B9D'
    },
    { 
      value: new Set(sessions.map(s => s.photographer)).size, 
      label: 'Фотографов',
      icon: 'icons/stat-photographer.svg',
      color: '#C44569'
    },
    { 
      value: sessions.length > 0 
        ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
        : 0, 
      label: 'Средняя оценка',
      icon: 'icons/stat-rating.svg',
      color: '#F59E0B'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <div class="stat-icon-wrapper" style="background: ${stat.color}15;">
          <img src="${stat.icon}" alt="" width="36" height="36" />
        </div>
        <div class="stat-content">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
}

function initBookPage() {
  const bookingForm = document.getElementById('bookingForm')
  const stylesGrid = document.getElementById('stylesGrid')
  const dateInput = document.getElementById('date')
  const successMessage = document.getElementById('successMessage')
  const photographersGrid = document.getElementById('photographersGrid')
  const advantagesGrid = document.getElementById('advantagesGrid')

  let selectedStyle = null

  if (stylesGrid) {
    stylesGrid.innerHTML = styles.map(style => `
      <button type="button" class="style-card" data-id="${style.id}" style="border-color: rgba(255, 107, 157, 0.2);">
        <div class="style-icon-wrapper" style="background: ${style.color}20;">
          <img src="${style.icon}" alt="${style.name}" width="40" height="40" style="filter: drop-shadow(0 2px 4px ${style.color}30);" />
        </div>
        <div class="style-content">
          <h3 class="style-name">${style.name}</h3>
          <p class="style-description">${style.description}</p>
          <div class="style-meta">
            <span class="style-duration">${style.duration}</span>
            <span class="style-price">${style.price}</span>
          </div>
        </div>
      </button>
    `).join('')

    stylesGrid.querySelectorAll('.style-card').forEach(button => {
      button.addEventListener('click', function() {
        stylesGrid.querySelectorAll('.style-card').forEach(btn => {
          btn.classList.remove('selected')
          btn.style.borderColor = 'rgba(255, 107, 157, 0.2)'
          const check = btn.querySelector('.style-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        this.style.borderColor = styles.find(s => s.id === this.dataset.id).color
        selectedStyle = styles.find(s => s.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'style-check'
        check.style.background = selectedStyle.color + '15'
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="24" height="24" style="filter: drop-shadow(0 2px 4px ${selectedStyle.color}30);" />`
        this.appendChild(check)
      })
    })
  }

  if (photographersGrid) {
    photographersGrid.innerHTML = photographers.map(photographer => {
      const stars = Array.from({ length: 5 }, () => 
        `<img src="icons/stat-rating.svg" alt="star" width="16" height="16" style="filter: drop-shadow(0 2px 4px rgba(245, 158, 11, 0.3));" />`
      ).join('')
      
      return `
        <div class="photographer-card">
          <div class="photographer-avatar">
            <img src="icons/stat-photographer.svg" alt="" width="40" height="40" />
          </div>
          <h3 class="photographer-name">${photographer.name}</h3>
          <p class="photographer-experience">${photographer.experience} опыта</p>
          <div class="photographer-rating">
            ${stars}
          </div>
          <p class="photographer-specialty">${photographer.specialty}</p>
        </div>
      `
    }).join('')
  }

  if (advantagesGrid) {
    advantagesGrid.innerHTML = advantages.map(advantage => `
      <div class="advantage-card">
        <div class="advantage-icon">
          <img src="${advantage.icon}" alt="" width="32" height="32" />
        </div>
        <h3 class="advantage-title">${advantage.title}</h3>
        <p class="advantage-text">${advantage.text}</p>
      </div>
    `).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (bookingForm) {
    bookingForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedStyle && dateInput.value) {
        const savedSessions = localStorage.getItem('photoSessions')
        const sessions = savedSessions ? JSON.parse(savedSessions) : []
        
        const randomPhotographer = photographers[Math.floor(Math.random() * photographers.length)]
        const newSession = {
          id: Date.now(),
          photographer: randomPhotographer.name,
          theme: selectedStyle.name,
          date: dateInput.value,
          rating: 0
        }

        sessions.push(newSession)
        localStorage.setItem('photoSessions', JSON.stringify(sessions))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedStyle = null
          dateInput.value = ''
          stylesGrid.querySelectorAll('.style-card').forEach(btn => {
            btn.classList.remove('selected')
            btn.style.borderColor = 'rgba(255, 107, 157, 0.2)'
            const check = btn.querySelector('.style-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  slides[currentSlide].classList.remove('active')
  dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

